"use strict";
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
Object.defineProperty(exports, "__esModule", { value: true });
// The module 'vscode' contains the VS Code extensibility API
// Import the module and reference it with the alias vscode in your code below
const vscode = require("vscode");
const path_1 = require("path");
const fs = require("fs");
const cp = require("child_process");
// import * as request from 'request';
const request = require("dropin-request");
let PREVIEW_URL = `https://${process.env.TOOL_HOST}/proxy/8086`;
function delay(ms) {
    return __awaiter(this, void 0, void 0, function* () {
        return new Promise(resolve => setTimeout(resolve, ms));
    });
}
function prepare_running() {
    try {
        cp.execSync('pgrep -f prepare');
        return true;
    }
    catch (_a) {
        return false;
    }
}
function prepare_progress(preparing, prepare_log, progress) {
    return __awaiter(this, void 0, void 0, function* () {
        let state = 0;
        while (prepare_running()) {
            let data = fs.readFileSync(preparing, 'utf8').toString();
            if (/Executing transaction/.test(data)) {
                if (state < 80) {
                    progress.report({ increment: 20, message: "Executing transaction" });
                    state = state + 20;
                }
            }
            else if (/Verifying transaction/.test(data)) {
                if (state < 60) {
                    progress.report({ increment: 20, message: "Verifying transaction" });
                    state = state + 20;
                }
            }
            else if (/Downloading and Extracting/.test(data)) {
                if (state < 40) {
                    progress.report({ increment: 20, message: "Downloading and Extracting" });
                    state = state + 20;
                }
            }
            else if (/Solving environment/.test(data)) {
                if (state < 20) {
                    progress.report({ increment: 20, message: "Solving environment" });
                    state = state + 20;
                }
            }
            yield delay(500);
        }
    });
}
function getCommands(dir) {
    const cmds = cp.execSync(`/opt/continuum/anaconda/envs/lab_launch/bin/anaconda-project list-commands --directory ${dir} | sed '1,/^=/d' | awk '{print $1}'`);
    return cmds.toString().trim().split('\n');
}
function runCommand(command, dir, progress, timeout = 20000) {
    const output = vscode.window.createOutputChannel(`anaconda-project run ${command}`);
    const tool_host = process.env.TOOL_HOST;
    output.show();
    output.appendLine(`## anaconda-project run ${command} --anaconda-project-port 8086 --anaconda-project-address 0.0.0.0 --anaconda-project-host $TOOL_HOST`);
    let proc = cp.spawn(`/opt/continuum/anaconda/envs/lab_launch/bin/anaconda-project`, ['run', '--directory', dir.toString(),
        command,
        '--anaconda-project-port', '8086',
        '--anaconda-project-address', '0.0.0.0',
        '--anaconda-project-host', tool_host
    ], {
        shell: true
    });
    proc.stderr.on('data', (data) => {
        output.appendLine(data.toString().trim());
    });
    proc.stdout.on('data', (data) => {
        output.appendLine(data.toString().trim());
    });
    proc.on('close', (code, signal) => {
        if (code == 1 && signal == null) {
            progress.report({ increment: 100 });
            vscode.window.showErrorMessage(`The command ${command} failed to run.`);
            setTimeout(output.dispose, 5000); //don't trash it too early
        }
        else {
            output.dispose();
        }
    });
    return proc;
}
function wait_for_microservice(proc, timeout = 20000) {
    return new Promise((resolve, reject) => {
        let timer = 0;
        function poller() {
            //This must always be http://localhost:8086.
            //We don't have auth headers so we cannot poll on the PREVIEW_URL
            request.get('http://localhost:8086', { timeout: timeout })
                .on('error', (err) => {
                //this is the connection refused error
                if (timer <= timeout) {
                    timer += 500;
                    setTimeout(poller, 500);
                }
                else {
                    reject(false);
                }
            })
                .on('response', (res) => {
                console.log(res.statusCode);
                if (res.statusCode == 200) {
                    console.log(res.read());
                    resolve(true);
                }
                else {
                    if (timer <= timeout) {
                        timer += 500;
                        setTimeout(poller, 500);
                    }
                    else {
                        reject(false);
                    }
                }
            });
        }
        poller();
    });
}
// this method is called when your extension is activated
function activate(context) {
    // Use the console to output diagnostic information (console.log) and errors (console.error)
    // This line of code will only be executed once when your extension is activated
    console.log('Activating "ae5-session"');
    if (!vscode.workspace.workspaceFolders) {
        return vscode.window.showWarningMessage('No folder or workspace opened');
    }
    const projectFolder = vscode.workspace.workspaceFolders[0].uri;
    let disposable = vscode.commands.registerCommand('ae5-session.preview', () => {
        let commands = getCommands(projectFolder.path);
        vscode.window.showQuickPick(commands, { placeHolder: 'Choose command to run' }).then(command => {
            if (command) {
                vscode.window.withProgress({
                    location: vscode.ProgressLocation.Notification,
                    title: 'Anaconda Enterprise',
                    cancellable: true
                }, (progress, token) => {
                    return new Promise((resolve, reject) => {
                        progress.report({ message: `anaconda-project run ${command} --anaconda-project-port 8086 --anaconda-project-address 0.0.0.0 --anaconda-project-host $TOOL_HOST` });
                        let proc = runCommand(command, projectFolder.path, progress);
                        token.onCancellationRequested(() => {
                            proc.kill();
                            console.log("User canceled the long running operation");
                        });
                        wait_for_microservice(proc).then(value => {
                            console.log(`resolved: ${value}`);
                            progress.report({ increment: 100, message: "Finished!" });
                            const preview = vscode.window.createWebviewPanel('deploymentCMD', 'Deployment preview', vscode.ViewColumn.One, {
                                enableScripts: true,
                                retainContextWhenHidden: false,
                            });
                            preview.onDidDispose(() => { proc.kill('SIGINT'); }, null, context.subscriptions);
                            preview.webview.html = `
							<html>
								<body style="margin:0px;padding:0px;overflow:hidden">
									<div style="position:fixed;height:100%;width:100%;">
										<iframe src="${PREVIEW_URL}" frameborder="0" style="overflow:hidden;height:100%;width:100%" height="100%" width="100%"></iframe>
									</div>
								</body>
							</html>`;
                            vscode.window.showInformationMessage('The command is running. Close preview tab to terminate.', 'Terminate', 'Dismiss').then(response => {
                                if (response == 'Terminate') {
                                    preview.dispose();
                                }
                            });
                            resolve();
                        }).catch(err => {
                            progress.report({ increment: 100, message: "Error" });
                            console.log(`oops ${err}`);
                            setTimeout(() => {
                                reject();
                            }, 3000);
                            vscode.window.showErrorMessage(`The command ${command} failed. Check that the deployment is served on port 8086.`, 'Dismiss');
                            proc.kill();
                        });
                    });
                });
            }
        });
    });
    context.subscriptions.push(disposable);
    const preparing = projectFolder.with({ path: path_1.posix.join(projectFolder.path, '..', 'preparing') });
    const prepare_log = projectFolder.with({ path: path_1.posix.join(projectFolder.path, '..', 'prepare.log') });
    const json_errors = projectFolder.with({ path: path_1.posix.join(projectFolder.path, '..', '.vscode', 'SETTINGS_PARSE_ERROR') });
    if (fs.existsSync(json_errors.path)) {
        let contents = fs.readFileSync(json_errors.path, 'utf8').toString().trim();
        vscode.window.showErrorMessage(contents, 'Dismiss');
    }
    if (fs.existsSync(preparing.path)) {
        var tail;
        vscode.window.showInformationMessage('anaconda-project prepare is running...', 'Watch output', 'Dismiss').then(response => {
            if (response == 'Watch output') {
                tail = vscode.window.createTerminal('tail -f ../preparing');
                tail.sendText('tail -f ../preparing');
                tail.show();
            }
        });
        vscode.window.withProgress({
            location: vscode.ProgressLocation.Notification,
            title: 'Anaconda Enterprise',
            cancellable: false
        }, (progress) => __awaiter(this, void 0, void 0, function* () {
            progress.report({ message: 'Preparing environment' });
            yield delay(2000);
            yield prepare_progress(preparing.path, prepare_log.path, progress);
            progress.report({ increment: 100, message: "Finished!" });
            if (tail) {
                setTimeout(() => {
                    tail.sendText('\x03');
                    tail.dispose();
                }, 5000);
            }
            var p = new Promise(resolve => {
                setTimeout(() => {
                    resolve();
                }, 2000);
                if (fs.existsSync(prepare_log.path)) {
                    let pythonExtension = vscode.extensions.getExtension('ms-python.python');
                    if (pythonExtension && pythonExtension.isActive) {
                        vscode.window.showInformationMessage('The project environment is ready. You must reload VSCode to utilize the environment.', 'Reload VSCode', 'Dismiss').then(response => {
                            if (response == 'Reload VSCode') {
                                vscode.commands.executeCommand("workbench.action.reloadWindow");
                            }
                        });
                    }
                    else {
                        return vscode.window.showInformationMessage(`The project environment is ready.`, 'Show output', 'Dismiss').then(response => {
                            if (response == 'Show output') {
                                const output = vscode.window.createOutputChannel("Environment Preparation");
                                output.append(fs.readFileSync(prepare_log.path, 'utf8').toString());
                                output.show();
                            }
                        });
                    }
                }
                else {
                    vscode.window.showErrorMessage('There was an error creating the environment.', 'Show errors', 'Dismiss').then(response => {
                        if (response == 'Show errors') {
                            const output = vscode.window.createOutputChannel("Environment Preparation Errors");
                            output.append(fs.readFileSync(preparing.path, 'utf8').toString());
                            output.show();
                        }
                    });
                }
            });
            return p;
        }));
    }
    else if (!fs.existsSync(prepare_log.path)) {
        return vscode.window.showWarningMessage('No preparing or prepare.log files.');
    }
}
exports.activate = activate;
// this method is called when your extension is deactivated
function deactivate() { }
exports.deactivate = deactivate;
//# sourceMappingURL=extension.js.map